#!/usr/bin/env node
/**
 * Caring AI Server — Interactive Setup
 * Downloads llama-server binary and default model.
 * Run: node src/setup.js [--model phi-3.5-mini|llama-3.1-8b]
 */
import { ensureLlamaServer, ensureModel, listModels, hasLlamaServer, hasModel } from './model-manager.js';
import { load } from './config.js';

const args = process.argv.slice(2);
const modelArg = args.find((a) => a.startsWith('--model='))?.split('=')[1]
  || (args.indexOf('--model') >= 0 ? args[args.indexOf('--model') + 1] : null);

async function run() {
  const cfg = load();
  const modelId = modelArg || cfg.defaultModel;

  console.log('╔══════════════════════════════════════╗');
  console.log('║      Caring AI — Setup            ║');
  console.log('╚══════════════════════════════════════╝');
  console.log();

  // Step 1: llama-server binary
  if (hasLlamaServer()) {
    console.log('✓ llama-server binary already downloaded');
  } else {
    console.log('↓ Downloading llama-server binary...');
    await ensureLlamaServer((p) => {
      if (p.percent >= 0) {
        process.stdout.write(`\r  ${p.percent}% (${formatBytes(p.downloaded)} / ${formatBytes(p.total)})`);
      }
    });
    console.log('\n✓ llama-server binary ready');
  }

  // Step 2: Model
  if (hasModel(modelId)) {
    console.log(`✓ Model ${modelId} already downloaded`);
  } else {
    const model = cfg.models[modelId];
    if (!model) {
      console.error(`✗ Unknown model: ${modelId}`);
      console.log('Available:', Object.keys(cfg.models).join(', '));
      process.exit(1);
    }
    console.log(`↓ Downloading model ${modelId} (${model.size})...`);
    await ensureModel(modelId, (p) => {
      if (p.percent >= 0) {
        process.stdout.write(`\r  ${p.percent}% (${formatBytes(p.downloaded)} / ${formatBytes(p.total)})`);
      }
    });
    console.log(`\n✓ Model ${modelId} ready`);
  }

  console.log();
  console.log('Setup complete! Start the server with:');
  console.log('  npm start');
  console.log();
  console.log('Available models:');
  for (const m of listModels()) {
    console.log(`  ${m.downloaded ? '✓' : '○'} ${m.displayName || m.id} (${m.size}) — ${m.description}`);
  }
}

function formatBytes(bytes) {
  if (bytes < 1024) return bytes + ' B';
  if (bytes < 1048576) return (bytes / 1024).toFixed(1) + ' KB';
  if (bytes < 1073741824) return (bytes / 1048576).toFixed(1) + ' MB';
  return (bytes / 1073741824).toFixed(2) + ' GB';
}

run().catch((err) => {
  console.error('Setup failed:', err.message);
  process.exit(1);
});
