/**
 * Caring AI Server — Model & Binary Manager
 * Downloads llama-server binary and GGUF models.
 */
import { existsSync, statSync, createWriteStream, unlinkSync, chmodSync } from 'fs';
import { join } from 'path';
import { pipeline } from 'stream/promises';
import { MODELS_DIR, BIN_DIR, load } from './config.js';

// llama.cpp release binary for macOS ARM64
const LLAMA_SERVER_URL = 'https://github.com/ggml-org/llama.cpp/releases/latest/download/llama-server-macos-arm64';
const LLAMA_SERVER_BIN = join(BIN_DIR, 'llama-server');

/**
 * Check if llama-server binary exists
 */
export function hasLlamaServer() {
  return existsSync(LLAMA_SERVER_BIN);
}

export function getLlamaServerPath() {
  return LLAMA_SERVER_BIN;
}

/**
 * Download a file with progress callback
 * @param {string} url
 * @param {string} dest
 * @param {function} onProgress - ({downloaded, total, percent, label}) => void
 */
async function downloadFile(url, dest, onProgress) {
  const tmp = dest + '.downloading';
  // Follow redirects (HuggingFace, GitHub)
  const res = await fetch(url, { redirect: 'follow' });
  if (!res.ok) throw new Error(`Download failed: ${res.status} ${res.statusText} — ${url}`);

  const total = parseInt(res.headers.get('content-length') || '0', 10);
  let downloaded = 0;

  const fileStream = createWriteStream(tmp);
  const reader = res.body.getReader();

  while (true) {
    const { done, value } = await reader.read();
    if (done) break;
    fileStream.write(value);
    downloaded += value.length;
    if (onProgress) {
      onProgress({
        downloaded,
        total,
        percent: total > 0 ? Math.round((downloaded / total) * 100) : -1,
      });
    }
  }

  fileStream.end();
  await new Promise((resolve, reject) => {
    fileStream.on('finish', resolve);
    fileStream.on('error', reject);
  });

  // Rename tmp → dest
  const { renameSync } = await import('fs');
  renameSync(tmp, dest);
}

/**
 * Download llama-server binary if missing
 */
export async function ensureLlamaServer(onProgress) {
  if (hasLlamaServer()) return LLAMA_SERVER_BIN;

  console.log('Downloading llama-server binary...');
  await downloadFile(LLAMA_SERVER_URL, LLAMA_SERVER_BIN, (p) => {
    if (onProgress) onProgress({ ...p, label: 'llama-server' });
  });
  chmodSync(LLAMA_SERVER_BIN, 0o755);
  console.log('llama-server downloaded.');
  return LLAMA_SERVER_BIN;
}

/**
 * Get path to a model GGUF file
 */
export function getModelPath(modelId) {
  const cfg = load();
  const model = cfg.models[modelId];
  if (!model) throw new Error(`Unknown model: ${modelId}`);
  return join(MODELS_DIR, model.file);
}

/**
 * Check if a model is downloaded
 */
export function hasModel(modelId) {
  try {
    const path = getModelPath(modelId);
    return existsSync(path);
  } catch {
    return false;
  }
}

/**
 * Download a model if missing
 */
export async function ensureModel(modelId, onProgress) {
  const cfg = load();
  const model = cfg.models[modelId];
  if (!model) throw new Error(`Unknown model: ${modelId}`);

  const dest = join(MODELS_DIR, model.file);
  if (existsSync(dest)) {
    // Verify size roughly matches
    const stat = statSync(dest);
    if (stat.size > model.sizeBytes * 0.9) return dest;
    // Incomplete download — remove and retry
    unlinkSync(dest);
  }

  console.log(`Downloading model ${modelId} (${model.size})...`);
  await downloadFile(dest, dest, (p) => {
    if (onProgress) onProgress({ ...p, label: modelId });
  });
  console.log(`Model ${modelId} ready.`);
  return dest;
}

/**
 * List all models with download status
 */
export function listModels() {
  const cfg = load();
  return Object.entries(cfg.models).map(([id, model]) => ({
    id,
    ...model,
    downloaded: hasModel(id),
  }));
}
