/**
 * Caring AI Server — llama-server Process Manager
 * Spawns and manages the llama-server child process.
 */
import { spawn } from 'child_process';
import { EventEmitter } from 'events';
import { getLlamaServerPath, getModelPath, hasLlamaServer, hasModel } from './model-manager.js';
import { load } from './config.js';

class LlamaProcess extends EventEmitter {
  constructor() {
    super();
    this.process = null;
    this.currentModel = null;
    this.state = 'stopped'; // stopped | starting | running | error
    this.lastError = null;
  }

  /**
   * Start llama-server with the given model
   */
  async start(modelId) {
    if (this.state === 'running' && this.currentModel === modelId) return;
    if (this.state === 'running') await this.stop();

    const cfg = load();
    modelId = modelId || cfg.defaultModel;

    if (!hasLlamaServer()) throw new Error('llama-server binary not found. Run setup first.');
    if (!hasModel(modelId)) throw new Error(`Model ${modelId} not downloaded. Run setup first.`);

    const binPath = getLlamaServerPath();
    const modelPath = getModelPath(modelId);

    this.state = 'starting';
    this.currentModel = modelId;
    this.emit('state', this.state);

    const args = [
      '--model', modelPath,
      '--port', String(cfg.llamaPort),
      '--host', '127.0.0.1',
      '--ctx-size', String(cfg.contextSize),
      '--n-gpu-layers', String(cfg.gpuLayers),
      '--threads', String(Math.max(1, (await import('os')).then ? 4 : 4)),
      // OpenAI-compatible API
      '--chat-template', 'chatml',
    ];

    console.log(`Starting llama-server: ${modelId}`);
    this.process = spawn(binPath, args, {
      stdio: ['ignore', 'pipe', 'pipe'],
      env: { ...process.env },
    });

    this.process.stdout.on('data', (data) => {
      const line = data.toString().trim();
      if (line) console.log(`[llama] ${line}`);
      // Detect when server is ready
      if (line.includes('server is listening on') || line.includes('main: server is listening')) {
        this.state = 'running';
        this.emit('state', this.state);
        console.log(`llama-server ready (${modelId})`);
      }
    });

    this.process.stderr.on('data', (data) => {
      const line = data.toString().trim();
      if (line) console.log(`[llama:err] ${line}`);
      // llama.cpp logs progress to stderr
      if (line.includes('server is listening on') || line.includes('main: server is listening')) {
        this.state = 'running';
        this.emit('state', this.state);
        console.log(`llama-server ready (${modelId})`);
      }
    });

    this.process.on('error', (err) => {
      this.state = 'error';
      this.lastError = err.message;
      this.emit('state', this.state);
      console.error('llama-server process error:', err);
    });

    this.process.on('exit', (code) => {
      if (this.state !== 'stopped') {
        this.state = code === 0 ? 'stopped' : 'error';
        this.lastError = code !== 0 ? `Exited with code ${code}` : null;
        this.emit('state', this.state);
      }
      this.process = null;
    });

    // Wait for ready with timeout
    await new Promise((resolve, reject) => {
      const timeout = setTimeout(() => {
        if (this.state !== 'running') {
          this.state = 'error';
          this.lastError = 'Startup timeout (30s)';
          this.emit('state', this.state);
          resolve(); // Don't reject — let caller check state
        }
      }, 30_000);

      this.once('state', (s) => {
        if (s === 'running' || s === 'error') {
          clearTimeout(timeout);
          resolve();
        }
      });
    });
  }

  /**
   * Stop the llama-server process
   */
  async stop() {
    if (!this.process) {
      this.state = 'stopped';
      return;
    }
    this.state = 'stopped';
    this.emit('state', this.state);
    this.process.kill('SIGTERM');

    // Wait for exit
    await new Promise((resolve) => {
      const timeout = setTimeout(() => {
        if (this.process) this.process.kill('SIGKILL');
        resolve();
      }, 5000);
      if (this.process) {
        this.process.once('exit', () => {
          clearTimeout(timeout);
          resolve();
        });
      } else {
        clearTimeout(timeout);
        resolve();
      }
    });
    this.process = null;
    this.currentModel = null;
  }

  /**
   * Get current status
   */
  status() {
    return {
      state: this.state,
      model: this.currentModel,
      error: this.lastError,
      pid: this.process?.pid || null,
    };
  }
}

export default LlamaProcess;
