/**
 * Caring AI Server
 * 
 * Exposes an OpenAI-compatible API on localhost:11435
 * Manages llama-server process and model downloads.
 * 
 * Extension connects here for fast, private, local AI.
 */
import express from 'express';
import { fileURLToPath } from 'url';
import { dirname, join } from 'path';
import { load, save } from './config.js';
import LlamaProcess from './llama-process.js';
import {
  ensureLlamaServer,
  ensureModel,
  listModels,
  hasLlamaServer,
  hasModel,
} from './model-manager.js';

const __dirname = dirname(fileURLToPath(import.meta.url));

const app = express();
app.use(express.json({ limit: '10mb' }));

// Serve dashboard UI
app.use(express.static(join(__dirname, '..', 'public')));

// CORS for chrome-extension:// origins
app.use((req, res, next) => {
  const origin = req.headers.origin || '';
  if (
    origin.startsWith('chrome-extension://') ||
    origin === 'http://localhost' ||
    origin.startsWith('http://localhost:') ||
    origin.startsWith('http://127.0.0.1')
  ) {
    res.setHeader('Access-Control-Allow-Origin', origin);
    res.setHeader('Access-Control-Allow-Methods', 'GET, POST, OPTIONS');
    res.setHeader('Access-Control-Allow-Headers', 'Content-Type, Authorization');
  }
  if (req.method === 'OPTIONS') return res.sendStatus(204);
  next();
});

const llama = new LlamaProcess();

// Track active downloads
const downloads = new Map(); // key → {progress, label, status}

// ─── Health & Status ──────────────────────────────────────────────

app.get('/health', (req, res) => {
  res.json({ status: 'ok', version: '0.1.0' });
});

app.get('/v1/status', (req, res) => {
  const cfg = load();
  res.json({
    server: 'caring-ai',
    version: '0.1.0',
    llama: llama.status(),
    models: listModels(),
    config: {
      port: cfg.port,
      defaultModel: cfg.defaultModel,
      contextSize: cfg.contextSize,
    },
  });
});

// ─── Model Management ─────────────────────────────────────────────

app.get('/v1/models', (req, res) => {
  const models = listModels();
  // OpenAI-compatible format
  res.json({
    object: 'list',
    data: models.map((m) => ({
      id: m.id,
      object: 'model',
      owned_by: 'caring-consulting',
      name: m.displayName || m.id,
      displayName: m.displayName || m.id,
      description: m.description,
      size: m.size,
      downloaded: m.downloaded,
      capabilities: m.capabilities || [],
    })),
  });
});

app.post('/v1/models/:modelId/download', async (req, res) => {
  const { modelId } = req.params;
  const cfg = load();
  if (!cfg.models[modelId]) return res.status(404).json({ error: `Unknown model: ${modelId}` });

  if (hasModel(modelId)) return res.json({ status: 'already_downloaded' });

  if (downloads.has(modelId)) {
    return res.json({ status: 'downloading', ...downloads.get(modelId) });
  }

  downloads.set(modelId, { progress: 0, label: modelId, status: 'downloading' });
  res.json({ status: 'started' });

  // Download in background
  try {
    await ensureModel(modelId, (p) => {
      downloads.set(modelId, { ...p, status: 'downloading' });
    });
    downloads.set(modelId, { progress: 100, label: modelId, status: 'complete' });
  } catch (err) {
    downloads.set(modelId, { progress: 0, label: modelId, status: 'error', error: err.message });
  }
});

app.get('/v1/models/:modelId/download/status', (req, res) => {
  const { modelId } = req.params;
  if (downloads.has(modelId)) return res.json(downloads.get(modelId));
  if (hasModel(modelId)) return res.json({ status: 'complete', progress: 100 });
  res.json({ status: 'not_started' });
});

// ─── llama-server Lifecycle ───────────────────────────────────────

app.post('/v1/server/start', async (req, res) => {
  const { model } = req.body || {};
  try {
    await llama.start(model);
    res.json(llama.status());
  } catch (err) {
    res.status(500).json({ error: err.message });
  }
});

app.post('/v1/server/stop', async (req, res) => {
  await llama.stop();
  res.json({ status: 'stopped' });
});

app.post('/v1/server/switch', async (req, res) => {
  const { model } = req.body;
  if (!model) return res.status(400).json({ error: 'model required' });
  try {
    await llama.stop();
    await llama.start(model);
    res.json(llama.status());
  } catch (err) {
    res.status(500).json({ error: err.message });
  }
});

// ─── OpenAI-Compatible Chat Completions ───────────────────────────

app.post('/v1/chat/completions', async (req, res) => {
  if (llama.state !== 'running') {
    return res.status(503).json({
      error: {
        message: 'AI model not loaded. Start the server first.',
        type: 'server_error',
        code: 'model_not_loaded',
      },
    });
  }

  const cfg = load();
  const llamaUrl = `http://127.0.0.1:${cfg.llamaPort}/v1/chat/completions`;

  try {
    // Forward request to llama-server (it speaks OpenAI protocol)
    const llamaRes = await fetch(llamaUrl, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(req.body),
    });

    if (req.body.stream) {
      // Stream SSE back to client
      res.setHeader('Content-Type', 'text/event-stream');
      res.setHeader('Cache-Control', 'no-cache');
      res.setHeader('Connection', 'keep-alive');

      const reader = llamaRes.body.getReader();
      const decoder = new TextDecoder();

      while (true) {
        const { done, value } = await reader.read();
        if (done) break;
        res.write(decoder.decode(value, { stream: true }));
      }
      res.end();
    } else {
      const data = await llamaRes.json();
      res.json(data);
    }
  } catch (err) {
    res.status(502).json({
      error: {
        message: `llama-server error: ${err.message}`,
        type: 'server_error',
      },
    });
  }
});

// ─── Setup Endpoint (download binary + default model) ─────────────

app.post('/v1/setup', async (req, res) => {
  const cfg = load();
  const steps = [];

  try {
    if (!hasLlamaServer()) {
      steps.push({ step: 'llama-server', status: 'downloading' });
      await ensureLlamaServer();
      steps.push({ step: 'llama-server', status: 'done' });
    } else {
      steps.push({ step: 'llama-server', status: 'already_present' });
    }

    const modelId = req.body?.model || cfg.defaultModel;
    if (!hasModel(modelId)) {
      steps.push({ step: `model:${modelId}`, status: 'downloading' });
      await ensureModel(modelId);
      steps.push({ step: `model:${modelId}`, status: 'done' });
    } else {
      steps.push({ step: `model:${modelId}`, status: 'already_present' });
    }

    res.json({ status: 'ready', steps });
  } catch (err) {
    res.status(500).json({ status: 'error', error: err.message, steps });
  }
});

// ─── Config ───────────────────────────────────────────────────────

app.get('/v1/config', (req, res) => {
  const cfg = load();
  res.json({
    port: cfg.port,
    defaultModel: cfg.defaultModel,
    contextSize: cfg.contextSize,
    gpuLayers: cfg.gpuLayers,
  });
});

app.patch('/v1/config', (req, res) => {
  const cfg = load();
  const allowed = ['defaultModel', 'contextSize', 'gpuLayers'];
  for (const key of allowed) {
    if (req.body[key] !== undefined) cfg[key] = req.body[key];
  }
  save(cfg);
  res.json({ status: 'updated' });
});

// ─── Start Server ─────────────────────────────────────────────────

const cfg = load();
const PORT = cfg.port;

app.listen(PORT, cfg.host, () => {
  console.log(`Caring AI Server v0.1.0 listening on http://${cfg.host}:${PORT}`);
  console.log(`Endpoints:`);
  console.log(`  GET  /health                         — health check`);
  console.log(`  GET  /v1/status                      — full status`);
  console.log(`  GET  /v1/models                      — list models`);
  console.log(`  POST /v1/models/:id/download          — download a model`);
  console.log(`  POST /v1/server/start                 — start llama-server`);
  console.log(`  POST /v1/server/stop                  — stop llama-server`);
  console.log(`  POST /v1/chat/completions             — OpenAI-compatible chat`);
  console.log(`  POST /v1/setup                        — download binary + model`);

  // Auto-start if binary + default model exist
  if (hasLlamaServer() && hasModel(cfg.defaultModel)) {
    console.log(`Auto-starting with model: ${cfg.defaultModel}...`);
    llama.start(cfg.defaultModel).catch((err) => {
      console.error('Auto-start failed:', err.message);
    });
  } else {
    console.log('Run POST /v1/setup to download binary and model.');
  }
});

// Graceful shutdown
process.on('SIGINT', async () => {
  console.log('\nShutting down...');
  await llama.stop();
  process.exit(0);
});

process.on('SIGTERM', async () => {
  await llama.stop();
  process.exit(0);
});
