/**
 * Caring AI Server — Configuration
 */
import { homedir } from 'os';
import { join } from 'path';
import { existsSync, readFileSync, writeFileSync, mkdirSync } from 'fs';

const DATA_DIR = join(homedir(), '.caring-ai');
const CONFIG_PATH = join(DATA_DIR, 'config.json');
const MODELS_DIR = join(DATA_DIR, 'models');
const BIN_DIR = join(DATA_DIR, 'bin');

const DEFAULTS = {
  port: 11435,
  host: '127.0.0.1',
  // llama-server settings
  llamaPort: 11436,       // internal port for llama-server process
  contextSize: 4096,
  gpuLayers: -1,           // -1 = auto (use all available GPU layers)
  // Model defaults
  defaultModel: 'caring-lite',
  models: {
    'caring-lite': {
      file: 'Phi-3.5-mini-instruct-Q4_K_M.gguf',
      url: 'https://huggingface.co/bartowski/Phi-3.5-mini-instruct-GGUF/resolve/main/Phi-3.5-mini-instruct-Q4_K_M.gguf',
      size: '2.4 GB',
      sizeBytes: 2_576_980_480,
      description: 'Quick answers, summaries, and drafts',
      displayName: 'Caring AI Lite',
    },
    'caring-pro': {
      file: 'Qwen2.5-7B-Instruct-Q4_K_M.gguf',
      url: 'https://huggingface.co/Qwen/Qwen2.5-7B-Instruct-GGUF/resolve/main/qwen2.5-7b-instruct-q4_k_m.gguf',
      size: '4.7 GB',
      sizeBytes: 5_046_157_312,
      description: 'Advanced reasoning, chat, and smart actions',
      displayName: 'Caring AI Pro',
      capabilities: ['function_calling', 'structured_output', 'agentic'],
    },
    'caring-max': {
      file: 'Meta-Llama-3.1-8B-Instruct-Q4_K_M.gguf',
      url: 'https://huggingface.co/bartowski/Meta-Llama-3.1-8B-Instruct-GGUF/resolve/main/Meta-Llama-3.1-8B-Instruct-Q4_K_M.gguf',
      size: '4.9 GB',
      sizeBytes: 5_261_639_168,
      description: 'Maximum capability for complex tasks',
      displayName: 'Caring AI Max',
    },
  },
};

function ensureDirs() {
  for (const dir of [DATA_DIR, MODELS_DIR, BIN_DIR]) {
    if (!existsSync(dir)) mkdirSync(dir, { recursive: true });
  }
}

function load() {
  ensureDirs();
  if (!existsSync(CONFIG_PATH)) {
    save(DEFAULTS);
    return { ...DEFAULTS };
  }
  try {
    const stored = JSON.parse(readFileSync(CONFIG_PATH, 'utf8'));
    return { ...DEFAULTS, ...stored, models: { ...DEFAULTS.models, ...stored.models } };
  } catch {
    return { ...DEFAULTS };
  }
}

function save(cfg) {
  ensureDirs();
  writeFileSync(CONFIG_PATH, JSON.stringify(cfg, null, 2));
}

export { DATA_DIR, MODELS_DIR, BIN_DIR, CONFIG_PATH, DEFAULTS, load, save, ensureDirs };
